<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use App\Models\Message;
use App\Models\CompanyProfile;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class UserController extends Controller
{
    public function index(): JsonResponse
    {
        $users = User::with(['companyProfile.subscription'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $users,
        ], 200);
    }

    public function totalUsers(): JsonResponse
    {
        $totalUsers = User::count();
        $totalCandidates = User::where('User_type', 'candidate')->count();
        $totalCompanies = CompanyProfile::count(); // Assuming one company per CompanyProfile

        return response()->json([
            'status' => 'success',
            'total_users' => $totalUsers,
            'total_candidates' => $totalCandidates,
            'total_companies' => $totalCompanies,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $user = User::with(['companyProfile.subscription'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $user,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Email' => 'required|string|email|max:255|unique:users',
            'mobile_number' => 'sometimes|string|max:15|unique:users|regex:/^[0-9]{10,15}$/',
            'province' => 'sometimes|string|max:100',
            'latitude' => 'sometimes|numeric|between:-90,90',
            'longitude' => 'sometimes|numeric|between:-180,180',
            'has_used_trial' => 'sometimes|boolean',
            'Password' => 'required|string|min:8',
            'User_type' => 'required|in:candidate,company,super_admin',
            'Status' => 'sometimes|in:active,inactive,banned',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only(['Email', 'User_type', 'Status']);
        if ($request->has('mobile_number')) {
            $data['mobile_number'] = $request->mobile_number;
        }
        if ($request->has('province')) {
            $data['province'] = $request->province;
        }
        if ($request->has('latitude')) {
            $data['latitude'] = $request->latitude;
        }
        if ($request->has('longitude')) {
            $data['longitude'] = $request->longitude;
        }
        if ($request->has('has_used_trial')) {
            $data['has_used_trial'] = $request->has_used_trial;
        }
        $data['Password_hash'] = Hash::make($request->Password);

        $user = User::create($data);

        $roleName = ucfirst($request->User_type);
        $role = Role::where('Name', $roleName)->first();

        if ($role) {
            $user->roles()->detach();
            $user->roles()->attach($role->Role_id);
        } else {
            return response()->json([
                'status' => 'error',
                'message' => "Role for User_type '{$request->User_type}' not found",
            ], 422);
        }

        return response()->json([
            'status' => 'success',
            'data' => $user->load('roles'),
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $user = User::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Email' => 'sometimes|string|email|max:255|unique:users,Email,' . $user->User_id,
            'mobile_number' => 'sometimes|string|max:15|unique:users,mobile_number,' . $user->User_id . '|regex:/^[0-9]{10,15}$/',
            'province' => 'sometimes|string|max:100',
            'latitude' => 'sometimes|numeric|between:-90,90',
            'longitude' => 'sometimes|numeric|between:-180,180',
            'has_used_trial' => 'sometimes|boolean',
            'Password' => 'sometimes|string|min:8',
            'User_type' => 'sometimes|in:candidate,company,super_admin',
            'Status' => 'sometimes|in:active,inactive,banned',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only(['Email', 'User_type', 'Status']);
        if ($request->has('mobile_number')) {
            $data['mobile_number'] = $request->mobile_number;
        }
        if ($request->has('province')) {
            $data['province'] = $request->province;
        }
        if ($request->has('latitude')) {
            $data['latitude'] = $request->latitude;
        }
        if ($request->has('longitude')) {
            $data['longitude'] = $request->longitude;
        }
        if ($request->has('has_used_trial')) {
            $data['has_used_trial'] = $request->has_used_trial;
        }
        if ($request->has('Password')) {
            $data['Password_hash'] = Hash::make($request->Password);
        }

        if ($request->has('User_type')) {
            $roleName = ucfirst($request->User_type);
            $role = Role::where('Name', $roleName)->first();

            if ($role) {
                $user->roles()->detach();
                $user->roles()->attach($role->Role_id);
            } else {
                return response()->json([
                    'status' => 'error',
                    'message' => "Role for User_type '{$request->User_type}' not found",
                ], 422);
            }
        }

        $user->update($data);

        return response()->json([
            'status' => 'success',
            'data' => $user->load('roles'),
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $user = User::findOrFail($id);

        // Delete associated company profile if it exists
        if ($user->companyProfile) {
            $user->companyProfile()->delete();
        }

        // Delete associated messages
        Message::where('Sender_id', $id)->orWhere('Receiver_id', $id)->delete();

        $user->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'User deleted successfully',
        ], 200);
    }

    public function login(Request $request): JsonResponse
    {
        $request->validate([
            'identifier' => ['required', 'string', 'max:255'],
            'password' => ['required', 'string', 'min:8'],
        ]);

        $field = filter_var($request->identifier, FILTER_VALIDATE_EMAIL) ? 'Email' : 'mobile_number';
        $user = User::where($field, $request->identifier)->first();

        if (!$user || !Hash::check($request->password, $user->Password_hash)) {
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid credentials',
            ], 401);
        }

        $token = $user->createToken('auth_token')->plainTextToken;

        $profile = $user->candidateProfile ?? $user->companyProfile;
        $profileId = $profile ? $profile->getKey() : null;
        $profileData = $profile ? $profile->toArray() : null;

        // Eager load roles
        $user->load('roles');

        return response()->json([
            'status' => 'success',
            'data' => [
                'user' => [
                    'User_id' => $user->User_id,
                    'Email' => $user->Email,
                    'mobile_number' => $user->mobile_number,
                    'province' => $user->province,
                    'latitude' => $user->latitude,
                    'longitude' => $user->longitude,
                    'has_used_trial' => $user->has_used_trial,
                    'User_type' => $user->User_type,
                    'Status' => $user->Status,
                    'created_at' => $user->created_at,
                    'updated_at' => $user->updated_at,
                    'company_profile' => $user->company_profile,
                    'roles' => $user->roles->map(function ($role) {
                        return [
                            'Role_id' => $role->Role_id,
                            'Name' => $role->Name,
                            'Permissions' => $role->Permissions,
                        ];
                    })->all(),
                ],
                'token' => $token,
                'profile_id' => [
                    'id' => $profileId,
                    'data' => $profileData,
                ],
            ],
        ], 200);
    }

    public function webLogin(Request $request)
    {
        Log::info('Web login attempt: ' . json_encode($request->all()));
        $request->validate([
            'identifier' => ['required', 'string', 'max:255'],
            'password' => ['required', 'string', 'min:8'],
        ]);

        $field = filter_var($request->identifier, FILTER_VALIDATE_EMAIL) ? 'Email' : 'mobile_number';
        $user = User::where($field, $request->identifier)->first();

        if (!$user) {
            Log::warning('User not found for identifier: ' . $request->identifier);
            return back()->withErrors(['identifier' => 'User not found.']);
        }

        if (!Hash::check($request->password, $user->Password_hash)) {
            Log::warning('Password check failed for identifier: ' . $request->identifier);
            return back()->withErrors(['password' => 'Invalid credentials.']);
        }

        Log::info('Authentication successful for User_id: ' . $user->User_id);
        Session::put('user_id', $user->User_id);
        Session::put('user_type', $user->User_type);

        // Debug session
        Log::debug('Session after login: ' . json_encode(Session::all()));

        $redirectUrl = '/chat/' . $user->User_id;
        Log::info('Redirecting to: ' . $redirectUrl);

        return redirect($redirectUrl)->with('success', 'Login successful');
    }

    public function storeMessage(Request $request)
    {
        Log::info('Message store attempt: ' . json_encode($request->all()));
        $validator = Validator::make($request->all(), [
            'Sender_id' => 'required|exists:users,User_id',
            'Receiver_id' => 'required|exists:users,User_id',
            'Content' => 'required|string',
            'Is_read' => 'nullable|in:0,1,false,true',
        ]);

        if ($validator->fails()) {
            Log::error('Validation failed: ' . json_encode($validator->errors()));
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $message = new Message();
        $message->Sender_id = $request->Sender_id;
        $message->Receiver_id = $request->Receiver_id;
        $message->Content = $request->Content;
        $message->Is_read = filter_var($request->Is_read, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) ?? false;
        $message->save();

        event(new \App\Events\NewMessage($message));

        Log::info('Message stored successfully: ' . $message->Message_id);

        return response()->json([
            'status' => 'success',
            'message' => $message,
        ], 201);
    }

    public function getMessages(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'sender_id' => 'required|exists:users,User_id',
            'receiver_id' => 'required|exists:users,User_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $messages = Message::where(function ($query) use ($request) {
            $query->where('Sender_id', $request->sender_id)
                ->where('Receiver_id', $request->receiver_id);
        })->orWhere(function ($query) use ($request) {
            $query->where('Sender_id', $request->receiver_id)
                ->where('Receiver_id', $request->sender_id);
        })->orderBy('created_at', 'asc')->get();

        return response()->json([
            'status' => 'success',
            'data' => $messages,
        ], 200);
    }
}
